﻿using System;
using System.Reflection;
using System.Threading;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.VBECS.Communication.Channels;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages;

namespace gov.va.med.vbecs.DAL.VistALink.Client
{
    class VistaLinkPinger : IPinger
    {
        private ISequentialMessager _messager;

        /// <summary>
        /// Ping time out in milliseconds.
        /// </summary>
        private int _pingTimeout = 1000*60; // 1 minute by default

        /// <summary>
        /// This timer is used to send PingMessage messages to server periodically.
        /// </summary>
        private readonly Timer _pingTimer;

        // Logger
        private readonly ILogger _logger =
            LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);


        /// <summary>
        /// Constructor
        /// </summary>
        public VistaLinkPinger()
        {
            _pingTimer = new Timer(handle_ping_time_callback, null, 30000, 30000);
        }

        public void Start(IMessager messager)
        {
            _messager = (ISequentialMessager)messager;
            //Force ping message immediately
            _messager.Send(new VistALinkHeartbeatRequestMessage(), 30000,
                on_message_reseived_method);
        }

        private void on_message_reseived_method(IMessage message)
        {
            // check if it is error
            if (message is ErrorMessage)
            {
                _logger.Error(string.Format("Ping failed: {0}", message));
                _messager.UnderlyingMessager.Disconnect();
                return;
            }
            PingMessageReceived((IPingMessage) message);
        }

        public void Stop()
        {
            _pingTimer.Change(Timeout.Infinite, Timeout.Infinite);
        }

        public void PingMessageReceived(IPingMessage msg)
        {
            _logger.Debug("Ping message received");
            var oldTimeout = _pingTimeout;
            var pingResponse = (VistALinkHeartbeatResponseMessage)msg;
            _pingTimeout = pingResponse.HeartbeatRate.ToInt32Milliseconds();
            _logger.Debug(string.Format("Ping timeout was updated, old value: {0}, new value: {1}", oldTimeout, _pingTimeout));
        }

        private void handle_ping_time_callback(object state)
        {
            if (_messager == null || _messager.UnderlyingMessager == null || 
                _messager.UnderlyingMessager.CommunicationStatus != CommunicationStatus.Connected || _messager.IsBusy)
            {
                return;
            }

            try
            {
                var lastMinute = DateTime.Now.AddMinutes((double)_pingTimeout * -1/60000);
                if (_messager.LastReceivedTime > lastMinute ||
                    _messager.LastSentTime > lastMinute)
                {
                    return;
                }

                _logger.Debug("Send ping message");
                _messager.Send(new VistALinkHeartbeatRequestMessage(), 30000, on_message_reseived_method);
            }
            catch (Exception e)
            {
                _logger.Error(e.Message, e);
            }
        }
    }
}
